"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrdersService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const order_entity_1 = require("../../entities/order.entity");
const order_detail_entity_1 = require("../../entities/order-detail.entity");
const product_entity_1 = require("../../entities/product.entity");
const customer_entity_1 = require("../../entities/customer.entity");
const payment_entity_1 = require("../../entities/payment.entity");
let OrdersService = class OrdersService {
    orderRepo;
    orderDetailRepo;
    productRepo;
    customerRepo;
    dataSource;
    constructor(orderRepo, orderDetailRepo, productRepo, customerRepo, dataSource) {
        this.orderRepo = orderRepo;
        this.orderDetailRepo = orderDetailRepo;
        this.productRepo = productRepo;
        this.customerRepo = customerRepo;
        this.dataSource = dataSource;
    }
    generateOrderNumber() {
        const timestamp = new Date().getTime();
        const random = Math.floor(Math.random() * 1000);
        return `ORD-${timestamp}-${random}`;
    }
    toOrderResponseDto(order) {
        return {
            id: order.id,
            orderNumber: order.orderNumber,
            totalAmount: parseFloat(order.totalAmount.toString()),
            taxAmount: parseFloat(order.taxAmount.toString()),
            shippingCost: parseFloat(order.shippingCost.toString()),
            discountAmount: parseFloat(order.discountAmount.toString()),
            status: order.status,
            paymentStatus: order.paymentStatus,
            paymentMethod: order.paymentMethod,
            shippingAddress: order.shippingAddress,
            billingAddress: order.billingAddress,
            customerNotes: order.customerNotes,
            adminNotes: order.adminNotes,
            customerId: order.customerId,
            items: order.orderDetails?.map(detail => ({
                id: detail.id,
                productName: detail.productName,
                productImage: detail.productImage,
                quantity: detail.quantity,
                unitPrice: parseFloat(detail.unitPrice.toString()),
                totalPrice: parseFloat(detail.totalPrice.toString()),
                productId: detail.productId,
            })) || [],
            createdAt: order.createdAt,
            updatedAt: order.updatedAt,
            shippedAt: order.shippedAt,
            deliveredAt: order.deliveredAt,
            cancelledAt: order.cancelledAt,
        };
    }
    async create(createOrderDto, customerId) {
        const queryRunner = this.dataSource.createQueryRunner();
        await queryRunner.connect();
        await queryRunner.startTransaction();
        try {
            const customer = await this.customerRepo.findOne({ where: { id: Number(customerId) } });
            if (!customer) {
                throw new common_1.NotFoundException('Customer not found');
            }
            let totalAmount = 0;
            const orderDetails = [];
            for (const item of createOrderDto.items) {
                const product = await this.productRepo.findOne({
                    where: { id: item.productId },
                    relations: ['vendor']
                });
                if (!product) {
                    throw new common_1.NotFoundException(`Product with ID ${item.productId} not found`);
                }
                if (product.stock < item.quantity) {
                    throw new common_1.BadRequestException(`Insufficient stock for product: ${product.name}`);
                }
                const unitPrice = item.unitPrice || parseFloat(product.price.toString());
                const totalPrice = unitPrice * item.quantity;
                product.stock -= item.quantity;
                await queryRunner.manager.save(product_entity_1.Product, product);
                orderDetails.push({
                    productId: item.productId,
                    quantity: item.quantity,
                    unitPrice,
                    totalPrice,
                    productName: product.name,
                    productImage: product.image,
                });
                totalAmount += totalPrice;
            }
            const taxAmount = totalAmount * 0.1;
            const shippingCost = 9.99;
            const discountAmount = createOrderDto.couponCode ? totalAmount * 0.1 : 0;
            const finalTotal = totalAmount + taxAmount + shippingCost - discountAmount;
            const order = this.orderRepo.create({
                orderNumber: this.generateOrderNumber(),
                totalAmount: finalTotal,
                taxAmount,
                shippingCost,
                discountAmount,
                paymentMethod: createOrderDto.paymentMethod,
                shippingAddress: createOrderDto.shippingAddress,
                billingAddress: createOrderDto.billingAddress || createOrderDto.shippingAddress,
                customerNotes: createOrderDto.customerNotes,
                customerId: Number(customerId),
                orderDetails: orderDetails,
            });
            const savedOrder = await queryRunner.manager.save(order_entity_1.Order, order);
            await queryRunner.commitTransaction();
            const completeOrder = await this.orderRepo.findOne({
                where: { id: savedOrder.id ?? savedOrder[0]?.id },
                relations: ['orderDetails', 'customer'],
            });
            if (!completeOrder) {
                throw new common_1.NotFoundException(`Order with ID ${savedOrder.id} not found after creation`);
            }
            return this.toOrderResponseDto(completeOrder);
        }
        catch (error) {
            await queryRunner.rollbackTransaction();
            throw error;
        }
        finally {
            await queryRunner.release();
        }
    }
    async findAll() {
        const orders = await this.orderRepo.find({
            relations: ['orderDetails', 'customer'],
            order: { createdAt: 'DESC' },
        });
        return orders.map(order => this.toOrderResponseDto(order));
    }
    async findOne(id) {
        const order = await this.orderRepo.findOne({
            where: { id },
            relations: ['orderDetails', 'customer'],
        });
        if (!order) {
            throw new common_1.NotFoundException(`Order with ID ${id} not found`);
        }
        return this.toOrderResponseDto(order);
    }
    async findByCustomer(customerId) {
        const orders = await this.orderRepo.find({
            where: { customerId: Number(customerId) },
            relations: ['orderDetails'],
            order: { createdAt: 'DESC' },
        });
        return orders.map(order => this.toOrderResponseDto(order));
    }
    async updateStatus(id, updateOrderStatusDto) {
        const order = await this.orderRepo.findOne({ where: { id } });
        if (!order) {
            throw new common_1.NotFoundException(`Order with ID ${id} not found`);
        }
        if (updateOrderStatusDto.status) {
            order.status = updateOrderStatusDto.status;
            if (updateOrderStatusDto.status === order_entity_1.OrderStatus.SHIPPED && !order.shippedAt) {
                order.shippedAt = new Date();
            }
            else if (updateOrderStatusDto.status === order_entity_1.OrderStatus.DELIVERED && !order.deliveredAt) {
                order.deliveredAt = new Date();
            }
            else if (updateOrderStatusDto.status === order_entity_1.OrderStatus.CANCELLED && !order.cancelledAt) {
                order.cancelledAt = new Date();
            }
        }
        if (updateOrderStatusDto.paymentStatus) {
            order.paymentStatus = updateOrderStatusDto.paymentStatus;
        }
        if (updateOrderStatusDto.adminNotes) {
            order.adminNotes = updateOrderStatusDto.adminNotes;
        }
        const updatedOrder = await this.orderRepo.save(order);
        const completeOrder = await this.orderRepo.findOne({
            where: { id: updatedOrder.id },
            relations: ['orderDetails', 'customer'],
        });
        if (!completeOrder) {
            throw new common_1.NotFoundException(`Order with ID ${updatedOrder.id} not found after update`);
        }
        return this.toOrderResponseDto(completeOrder);
    }
    async cancelOrder(id, customerId) {
        const order = await this.orderRepo.findOne({ where: { id } });
        if (!order) {
            throw new common_1.NotFoundException(`Order with ID ${id} not found`);
        }
        if (customerId && String(order.customerId) !== customerId) {
            throw new common_1.ForbiddenException('You can only cancel your own orders');
        }
        if (order.status !== order_entity_1.OrderStatus.PENDING && order.status !== order_entity_1.OrderStatus.CONFIRMED) {
            throw new common_1.BadRequestException('Order cannot be cancelled at this stage');
        }
        order.status = order_entity_1.OrderStatus.CANCELLED;
        order.cancelledAt = new Date();
        const orderDetails = await this.orderDetailRepo.find({ where: { orderId: id } });
        for (const detail of orderDetails) {
            const product = await this.productRepo.findOne({ where: { id: detail.productId } });
            if (product) {
                product.stock += detail.quantity;
                await this.productRepo.save(product);
            }
        }
        const updatedOrder = await this.orderRepo.save(order);
        const completeOrder = await this.orderRepo.findOne({
            where: { id: updatedOrder.id },
            relations: ['orderDetails', 'customer'],
        });
        if (!completeOrder) {
            throw new common_1.NotFoundException(`Order with ID ${updatedOrder.id} not found after update`);
        }
        return this.toOrderResponseDto(completeOrder);
    }
    async getOrderStats() {
        const totalOrders = await this.orderRepo.count();
        const totalRevenue = await this.orderRepo
            .createQueryBuilder('order')
            .select('SUM(order.totalAmount)', 'total')
            .where('order.paymentStatus = :status', { status: payment_entity_1.PaymentStatus.PAID })
            .getRawOne();
        const statusCounts = await this.orderRepo
            .createQueryBuilder('order')
            .select('order.status', 'status')
            .addSelect('COUNT(*)', 'count')
            .groupBy('order.status')
            .getRawMany();
        const recentOrders = await this.orderRepo.find({
            relations: ['customer', 'orderDetails'],
            order: { createdAt: 'DESC' },
            take: 10,
        });
        return {
            totalOrders,
            totalRevenue: parseFloat(totalRevenue.total) || 0,
            statusCounts: statusCounts.reduce((acc, item) => {
                acc[item.status] = parseInt(item.count);
                return acc;
            }, {}),
            recentOrders: recentOrders.map(order => this.toOrderResponseDto(order)),
        };
    }
};
exports.OrdersService = OrdersService;
exports.OrdersService = OrdersService = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(order_entity_1.Order)),
    __param(1, (0, typeorm_1.InjectRepository)(order_detail_entity_1.OrderDetail)),
    __param(2, (0, typeorm_1.InjectRepository)(product_entity_1.Product)),
    __param(3, (0, typeorm_1.InjectRepository)(customer_entity_1.Customer)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.DataSource])
], OrdersService);
//# sourceMappingURL=orders.service.js.map